<?php

namespace App\Controllers\Guru;

use App\Controllers\BaseController;
use Dompdf\Dompdf;
use Dompdf\Options;

class Nilai extends BaseController
{
    protected $db;

    public function __construct()
    {
        $this->db = \Config\Database::connect();
    }

    public function index()
    {
        $guruId = session()->get('id');

        $jadwal = $this->db->table('jadwal_ujian')
            ->select('jadwal_ujian.*, kelas.nama_kelas, mapel.nama_mapel')
            ->join('kelas', 'kelas.id = jadwal_ujian.kelas_id')
            ->join('mapel', 'mapel.id = jadwal_ujian.mapel_id')
            ->where('jadwal_ujian.guru_id', $guruId)
            ->orderBy('jadwal_ujian.tanggal_ujian', 'DESC')
            ->get()->getResultArray();

        $data = [
            'title' => 'Rekap Nilai Ujian',
            'jadwal' => $jadwal
        ];

        return view('guru/nilai/index', $data);
    }

    public function detail($jadwalId)
    {
        $jadwal = $this->db->table('jadwal_ujian')
            ->select('jadwal_ujian.*, kelas.nama_kelas, mapel.nama_mapel')
            ->join('kelas', 'kelas.id = jadwal_ujian.kelas_id')
            ->join('mapel', 'mapel.id = jadwal_ujian.mapel_id')
            ->where('jadwal_ujian.id', $jadwalId)
            ->get()->getRowArray();

        $siswa = $this->db->table('siswa')
            ->select('siswa.id, siswa.nama_lengkap, siswa.nisn, status_ujian_siswa.*')
            ->join('status_ujian_siswa', 'status_ujian_siswa.siswa_id = siswa.id AND status_ujian_siswa.jadwal_id = ' . $jadwalId, 'left')
            ->where('siswa.kelas_id', $jadwal['kelas_id'])
            ->orderBy('siswa.nama_lengkap', 'ASC')
            ->get()->getResultArray();

        $data = [
            'title' => 'Detail Nilai: ' . $jadwal['nama_mapel'],
            'jadwal' => $jadwal,
            'siswa' => $siswa,
            'validation' => \Config\Services::validation()
        ];

        return view('guru/nilai/detail', $data);
    }

    public function simpanBobot()
    {
        $jadwalId = $this->request->getPost('jadwal_id');
        $bobotPg = $this->request->getPost('bobot_pg');
        $bobotPgKompleks = $this->request->getPost('bobot_pg_kompleks');
        $bobotBenarSalah = $this->request->getPost('bobot_benar_salah');
        $bobotEsai = $this->request->getPost('bobot_esai');

        $totalBobot = $bobotPg + $bobotPgKompleks + $bobotBenarSalah + $bobotEsai;

        if ($totalBobot != 100) {
            return redirect()->back()->with('error', 'Total persentase bobot harus pas 100%. Saat ini: ' . $totalBobot . '%');
        }

        $this->db->table('jadwal_ujian')->where('id', $jadwalId)->update([
            'bobot_pg' => $bobotPg,
            'bobot_pg_kompleks' => $bobotPgKompleks,
            'bobot_benar_salah' => $bobotBenarSalah,
            'bobot_esai' => $bobotEsai
        ]);

        $this->hitungUlangSemua($jadwalId);

        return redirect()->back()->with('success', 'Pengaturan bobot berhasil diperbarui & nilai siswa dihitung ulang.');
    }

    public function koreksi($jadwalId, $siswaId)
    {
        $siswa = $this->db->table('siswa')->where('id', $siswaId)->get()->getRowArray();
        
        // --- PERBAIKAN: Cek apakah siswa ditemukan ---
        if (!$siswa) {
            return redirect()->back()->with('error', 'Data siswa tidak ditemukan.');
        }

        $jadwal = $this->db->table('jadwal_ujian')->where('id', $jadwalId)->get()->getRowArray();
        
        // --- PERBAIKAN: Cek apakah jadwal ditemukan ---
        if (!$jadwal) {
            return redirect()->back()->with('error', 'Data jadwal ujian tidak ditemukan.');
        }

        $jawabanAll = $this->db->table('hasil_ujian')
            ->select('hasil_ujian.*, soal.pertanyaan, soal.kunci_jawaban, soal.jenis')
            ->join('soal', 'soal.id = hasil_ujian.soal_id')
            ->where('hasil_ujian.jadwal_id', $jadwalId)
            ->where('hasil_ujian.siswa_id', $siswaId)
            ->get()->getResultArray();

        $jawabanPg = [];
        $jawabanPgKompleks = [];
        $jawabanBenarSalah = [];
        $jawabanEsai = [];

        foreach ($jawabanAll as $j) {
            if ($j['jenis'] == 'pg') $jawabanPg[] = $j;
            elseif ($j['jenis'] == 'pg_kompleks') $jawabanPgKompleks[] = $j;
            elseif ($j['jenis'] == 'benar_salah') $jawabanBenarSalah[] = $j;
            elseif ($j['jenis'] == 'esai') $jawabanEsai[] = $j;
        }

        $hitungSkor = function($data, $jenis) {
            $benar = 0;
            $total = count($data);
            foreach ($data as $item) {
                if ($jenis == 'pg_kompleks') {
                    $kunci = json_decode($item['kunci_jawaban'], true);
                    $jawab = json_decode($item['jawaban_siswa'], true);
                    if (is_array($kunci) && is_array($jawab)) {
                        sort($kunci); sort($jawab);
                        if ($kunci === $jawab) $benar++;
                    }
                } else {
                    if (trim($item['jawaban_siswa']) == trim($item['kunci_jawaban'])) $benar++;
                }
            }
            return ['benar' => $benar, 'total' => $total, 'nilai' => ($total > 0 ? ($benar/$total)*100 : 0)];
        };

        $data = [
            'title' => 'Koreksi Jawaban: ' . $siswa['nama_lengkap'],
            'siswa' => $siswa,
            'jadwal' => $jadwal,
            'pg' => ['data' => $jawabanPg, 'stats' => $hitungSkor($jawabanPg, 'pg')],
            'kompleks' => ['data' => $jawabanPgKompleks, 'stats' => $hitungSkor($jawabanPgKompleks, 'pg_kompleks')],
            'bs' => ['data' => $jawabanBenarSalah, 'stats' => $hitungSkor($jawabanBenarSalah, 'benar_salah')],
            'esai' => $jawabanEsai
        ];

        return view('guru/nilai/koreksi', $data);
    }

    public function simpanKoreksi()
    {
        $jadwalId = $this->request->getPost('jadwal_id');
        $siswaId = $this->request->getPost('siswa_id');
        $nilaiKoreksiEsai = $this->request->getPost('nilai_esai'); 

        if ($nilaiKoreksiEsai) {
            foreach ($nilaiKoreksiEsai as $hasilId => $skor) {
                $this->db->table('hasil_ujian')->where('id', $hasilId)->update(['nilai_koreksi' => $skor]);
            }
        }

        $this->hitungNilaiPerSiswa($jadwalId, $siswaId);

        return redirect()->to("guru/nilai/detail/$jadwalId")->with('success', 'Nilai siswa berhasil disimpan.');
    }

    private function hitungNilaiPerSiswa($jadwalId, $siswaId)
    {
        $jadwal = $this->db->table('jadwal_ujian')->where('id', $jadwalId)->get()->getRowArray();
        
        $jawaban = $this->db->table('hasil_ujian')
            ->select('hasil_ujian.*, soal.jenis, soal.kunci_jawaban')
            ->join('soal', 'soal.id = hasil_ujian.soal_id')
            ->where('hasil_ujian.jadwal_id', $jadwalId)
            ->where('hasil_ujian.siswa_id', $siswaId)
            ->get()->getResultArray();

        $score = ['pg' => 0, 'pg_kompleks' => 0, 'benar_salah' => 0, 'esai' => 0];
        $total = ['pg' => 0, 'pg_kompleks' => 0, 'benar_salah' => 0, 'esai' => 0];

        foreach ($jawaban as $j) {
            $jenis = $j['jenis'];
            $total[$jenis]++;

            if ($jenis == 'esai') {
                $score['esai'] += $j['nilai_koreksi'];
            } elseif ($jenis == 'pg_kompleks') {
                $k = json_decode($j['kunci_jawaban'], true);
                $s = json_decode($j['jawaban_siswa'], true);
                if (is_array($k) && is_array($s)) {
                    sort($k); sort($s);
                    if ($k === $s) $score['pg_kompleks']++;
                }
            } else {
                if (trim($j['jawaban_siswa']) == trim($j['kunci_jawaban'])) $score[$jenis]++;
            }
        }

        $nilaiAkhir = [
            'pg' => ($total['pg'] > 0) ? ($score['pg'] / $total['pg']) * 100 : 0,
            'pg_kompleks' => ($total['pg_kompleks'] > 0) ? ($score['pg_kompleks'] / $total['pg_kompleks']) * 100 : 0,
            'benar_salah' => ($total['benar_salah'] > 0) ? ($score['benar_salah'] / $total['benar_salah']) * 100 : 0,
            'esai' => ($total['esai'] > 0) ? ($score['esai'] / $total['esai']) : 0, 
        ];

        $bobotTotal = 0;
        $totalSkorAkhir = 0;

        $types = [
            'pg' => 'bobot_pg', 
            'pg_kompleks' => 'bobot_pg_kompleks', 
            'benar_salah' => 'bobot_benar_salah', 
            'esai' => 'bobot_esai'
        ];

        foreach($types as $key => $column) {
            if ($total[$key] > 0) {
                $bobotTotal += $jadwal[$column];
                $totalSkorAkhir += ($nilaiAkhir[$key] * $jadwal[$column]);
            }
        }

        $finalScore = ($bobotTotal > 0) ? ($totalSkorAkhir / $bobotTotal) * 100 : 0; 
        if ($bobotTotal == 100) $finalScore = $totalSkorAkhir / 100;

        $this->db->table('status_ujian_siswa')
            ->where('jadwal_id', $jadwalId)
            ->where('siswa_id', $siswaId)
            ->update([
                'nilai_pg' => $nilaiAkhir['pg'],
                'nilai_pg_kompleks' => $nilaiAkhir['pg_kompleks'],
                'nilai_benar_salah' => $nilaiAkhir['benar_salah'],
                'nilai_esai' => $nilaiAkhir['esai'],
                'nilai_total' => $finalScore
            ]);
    }

    private function hitungUlangSemua($jadwalId)
    {
        $siswaIds = $this->db->table('status_ujian_siswa')
            ->select('siswa_id')
            ->where('jadwal_id', $jadwalId)
            ->get()->getResultArray();

        foreach ($siswaIds as $s) {
            $this->hitungNilaiPerSiswa($jadwalId, $s['siswa_id']);
        }
    }

    // --- UPDATE FITUR CETAK (PDF) ---
    public function cetak($jadwalId)
    {
        $guruId = session()->get('id');
        $guru = $this->db->table('guru')->where('id', $guruId)->get()->getRowArray();
        $sekolah = $this->db->table('sekolah')->where('id', 1)->get()->getRowArray();

        $jadwal = $this->db->table('jadwal_ujian')
            ->select('jadwal_ujian.*, kelas.nama_kelas, mapel.nama_mapel')
            ->join('kelas', 'kelas.id = jadwal_ujian.kelas_id')
            ->join('mapel', 'mapel.id = jadwal_ujian.mapel_id')
            ->where('jadwal_ujian.id', $jadwalId)
            ->get()->getRowArray();

        if (!$jadwal) {
            return redirect()->back()->with('error', 'Data ujian tidak ditemukan.');
        }

        $siswa = $this->db->table('siswa')
            ->select('siswa.nama_lengkap, siswa.nisn, status_ujian_siswa.*')
            ->join('status_ujian_siswa', 'status_ujian_siswa.siswa_id = siswa.id AND status_ujian_siswa.jadwal_id = ' . $jadwalId, 'left')
            ->where('siswa.kelas_id', $jadwal['kelas_id'])
            ->orderBy('siswa.nama_lengkap', 'ASC')
            ->get()->getResultArray();

        $data = [
            'guru' => $guru,
            'sekolah' => $sekolah,
            'jadwal' => $jadwal,
            'siswa' => $siswa
        ];

        // Generate PDF
        $options = new Options();
        $options->set('isRemoteEnabled', true);
        $options->set('defaultFont', 'Arial');

        $dompdf = new Dompdf($options);
        
        // Menggunakan view cetak yang diperbarui
        $html = view('guru/nilai/cetak', $data);
        
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        $filename = 'Nilai_' . str_replace(' ', '_', $jadwal['nama_mapel']) . '_' . str_replace(' ', '_', $jadwal['nama_kelas']);
        
        $dompdf->stream($filename, ["Attachment" => false]);
        exit();
    }
}